
/////////////////////// stdlib includes
#include <utility>

/////////////////////// Qt includes
#include <QObject>
#include <QtQml>
#include <QQmlEngine>
#include <QJSEngine>


/////////////////////// pappsomspp includes
// Needed because we use the pappso-specific JS class reference registration function
#include <pappsomspp/core/js_qml/pappsojsqml.h>


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/libXpertMassCore/XpertMassJsQml.hpp"

#include "MsXpS/libXpertMassGui/ScriptingEnginesBridge.hpp"
#include "MsXpS/libXpertMassGui/XpertMassGuiJsQml.hpp"

namespace MsXpS
{
namespace libXpertMassGui
{

ScriptingEnginesBridge::ScriptingEnginesBridge(QObject *parent)
  : QObject(parent), mp_qmlEngine(nullptr), mp_jsEngine(nullptr)
{
  initialize();
}

void
ScriptingEnginesBridge::initialize()
{
  if(mp_qmlEngine)
    delete mp_qmlEngine;

  mp_qmlEngine = new QQmlEngine(this);

  if(mp_jsEngine)
    delete mp_jsEngine;

  mp_jsEngine = new QJSEngine(this);

  QJSValue returned_js_value;
  // Make the bridge available to both engines
  exposeObject("scriptingEnginesBridge",
               "", /*alias*/
               this,
               returned_js_value,
               QJSEngine::ObjectOwnership::CppOwnership);
}

QJSValue
ScriptingEnginesBridge::evaluate(const QString &script)
{
  if(!mp_jsEngine)
    return QJSValue();
  return mp_jsEngine->evaluate(script);
}

QObject *
ScriptingEnginesBridge::createQmlObject(const QString &qml_code,
                                        QObject *parent)
{
  if(!mp_qmlEngine)
    return nullptr;

  QQmlComponent component(mp_qmlEngine);
  component.setData(qml_code.toUtf8(), QUrl());

  if(component.isError())
    {
      qWarning() << "QML Error:" << component.errors();
      return nullptr;
    }
  // Create with proper context and parent
  QObject *obj = component.create();
  if(obj)
    {
      if(parent)
        {
          obj->setParent(parent);
        }
      else
        {
          obj->setParent(this);
        }
      QJSValue returned_js_value;
      exposeObject(obj->objectName().isEmpty() ? "qmlObject"
                                               : obj->objectName(),
                   "", // alias
                   obj,
                   returned_js_value);
    }
  return obj;
}

QQmlEngine *
ScriptingEnginesBridge::getQmlEngine() const
{
  return mp_qmlEngine;
}

QJSEngine *
ScriptingEnginesBridge::getJsEngine() const
{
  return mp_jsEngine;
}

bool
ScriptingEnginesBridge::exposeObject(const QString &name,
                                     const QString &alias,
                                     QObject *object,
                                     QJSValue &returned_js_value,
                                     QJSEngine::ObjectOwnership ownership)
{
  // Reset error state
  m_lastError.clear();

  if(name.isEmpty())
    {
      m_lastError = "Object name cannot be empty";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }

  if(object == nullptr)
    {
      m_lastError = "Cannot expose null object";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }

  // Check thread affinity
  if(object->thread() != this->thread())
    {
      m_lastError =
        QString("Object belongs to different thread (object: %1, current: %2)")
          .arg(object->thread()
                 ? QString::number(reinterpret_cast<quintptr>(object->thread()))
                 : "null")
          .arg(this->thread()
                 ? QString::number(reinterpret_cast<quintptr>(this->thread()))
                 : "null");

      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }

  if(mp_qmlEngine)
    {
      // We need this switch because the parameter is QJSEngine::ObjectOwnership
      // and not QQmlEngine::ObjectOwnership.
      switch(ownership)
        {
          case QJSEngine::JavaScriptOwnership:
            QQmlEngine::setObjectOwnership(object,
                                           QQmlEngine::JavaScriptOwnership);
            break;
          case QJSEngine::ObjectOwnership::CppOwnership:
            QQmlEngine::setObjectOwnership(object, QQmlEngine::CppOwnership);
            break;
        }

      mp_qmlEngine->rootContext()->setContextProperty(name, object);
      if(!alias.isEmpty())
        mp_qmlEngine->rootContext()->setContextProperty(alias, object);
    }
  else
    {
      m_lastError = "No QML engine available";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }

  if(mp_jsEngine)
    {
      // Here, we do not need the switch above.
      returned_js_value = mp_jsEngine->newQObject(object);
      QJSEngine::setObjectOwnership(object, QJSEngine::JavaScriptOwnership);

      mp_jsEngine->globalObject().setProperty(name, returned_js_value);
      if(!alias.isEmpty())
        mp_jsEngine->globalObject().setProperty(alias, returned_js_value);
    }
  else
    {
      m_lastError = "No JavaScript engine available";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }

  return true;
}

bool
ScriptingEnginesBridge::exposeJsEngineJsValue(const QString &name,
                                              QJSValue &js_value)
{

  m_lastError.clear();

  if(mp_jsEngine == nullptr)
    {
      m_lastError = "No JavaScript engine available";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }
  else
    mp_jsEngine->globalObject().setProperty(name, js_value);

  return true;
}

QJSValue
ScriptingEnginesBridge::getExposedJsEngineObjectByName(const QString &name)
{
  if(mp_jsEngine == nullptr)
    {
      m_lastError = "No JavaScript engine available";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return false;
    }

  return mp_jsEngine->globalObject().property(name);
}

QObject *
ScriptingEnginesBridge::getExposedQmlEngineObjectByName(const QString &name)
{
  if(mp_qmlEngine == nullptr)
    {
      m_lastError = "No Qml engine available";
      emit errorOccurredSignal(m_lastError);
      qWarning() << m_lastError;
      return nullptr;
    }

  return mp_qmlEngine->rootContext()->objectForName(name);
}

//FIXME

// To add a namespace/classname to the registrar map, simply
// include pappsomspp/js_qml/jsclassregistrar.h and
// call REGISTER_JS_CLASS(namespace, classname) with no quotes.
// See at the bottom of any relevant class declaration (outside
// the namespace) for an example (Formula.hpp).
void
ScriptingEnginesBridge::registerJsConstructorForClassInRegistrarMap(
  const QString &name_space, const QString &class_name)
{
  qDebug() << "Registering JavaScript constructor for" << name_space
           << "::" << class_name;

  MsXpS::registerJsConstructorForNameSpaceClassNameInRegistrarMap(
    name_space, class_name, mp_jsEngine);
}

void
ScriptingEnginesBridge::registerJsConstructorForClassesInRegistrarMap(
  const std::vector<std::pair<QString, QString>> &namespace_class_pairs)
{
  for(const std::pair<QString, QString> &pair : namespace_class_pairs)
    registerJsConstructorForClassInRegistrarMap(pair.first, pair.second);

  qDebug() << "At this point, the registry contains"
           << MsXpS::getNameSpaceClassNameJsConstructorRegistrarMap().size()
           << "pairs";
}

void
ScriptingEnginesBridge::registerJsConstructorForEachClassInRegistrarMap()
{
  qDebug() << "Registering the constructor of the classes marked for JS.";

  // FIXME

  // The function below is located in pappsomspp src/js_qml/jsclassregistrar.h
  // in the global namespace (hence the :: before the function name)
  // All the classed deemed to be exposed to QJSEngine have a
  // REGISTER_JS_CLASS(namespace, classname)
  // in the global namespace right after the corresponding class
  // declaration(header file).
  MsXpS::registerJsConstructorForEachClassInRegistrarMap(mp_jsEngine);

  // Do the same for pappsomspp, but the namespace is pappso.
  MsXpS::registerJsConstructorForEachClassInRegistrarMap(mp_jsEngine);

  qDebug() << "Exiting function.";
}

void
ScriptingEnginesBridge::registerOwnJsEntities()
{
  qDebug() << "Entering function.";

  qDebug() << "Registering own entities for libXpertMass and libXpertMassGui.";

  MsXpS::libXpertMassCore::registerEnumsToQJSEngine(mp_jsEngine);
  MsXpS::libXpertMassCore::registerGlobalsToQJSEngine(mp_jsEngine);

  MsXpS::libXpertMassGui::registerEnumsToQJSEngine(mp_jsEngine);
  MsXpS::libXpertMassGui::registerGlobalsToQJSEngine(mp_jsEngine);

  qDebug() << "Exiting function.";
}


void
ScriptingEnginesBridge::registerExternalJsEntities(const QString &name_space)
{

  if(name_space == "pappsomspp")
    pappso::registerEnumsToQJSEngine(mp_jsEngine);
}

} // namespace libXpertMassGui
} // namespace MsXpS

