#ifndef SOUNDIO_H
#define SOUNDIO_H

#include <alsa/asoundlib.h>
#include "qsstvglobal.h"
#include <QStringList>
#include <QThread>
#include "utils/buffermanag.h"
#include "wavio.h"
#include "qsstvdefs.h"
#include <QTime>

/*! \file soundio.h */

/**
 * @brief soundio class
 *
 *General Info<br>
 * A frame is equivalent of one sample being played, irrespective of the number of channels or the number of bits.<br>
 * e.g.
    1 frame of a Stereo 48khz 16bit PCM stream is 4 bytes.
    1 frame of a 5.1 48khz 16bit PCM stream is 12 bytes.

A period is the number of frames in between each hardware interrupt. The poll() will return once a period.

The buffer is a ring buffer. The buffer size always has to be greater than one period size. Commonly this is 2*period size, but some hardware can do 8 periods per buffer. It is also possible for the buffer size to not be an integer multiple of the period size.

Now, if the hardware has been set to 48000Hz , 2 periods, of 1024 frames each, making a buffer size of 2048 frames. The hardware will interrupt 2 times per buffer. ALSA will endeavor to keep the buffer as full as possible. Once the first period of samples has been played, the third period of samples is transfered into the space the first one occupied while the second period of samples is being played. (normal ring buffer behaviour).

 */

#define CALIBRATIONLEADIN 100

class soundIO:public QThread
{
  Q_OBJECT
public:
  enum edataSrc{SNDINCARD,SNDINFILE,SNDINCARDTOFILE};
  enum edataDst{SNDOUTCARD,SNDOUTTOFILE};
  enum eplaybackState{PBINIT,PBSTARTING,PBRUNNING,PBCALIBRATE1,PBCALIBRATE2};
  enum ecaptureState{CPINIT,CPSTARTING,CPRUNNING,CPCALIBRATE,CPEND};
#define PERIODBITS 12
#define PERIODSIZE (1<<PERIODBITS)
#define BUFFERSIZE (8*PERIODSIZE)
  soundIO(int s=48000);
  void setSamplerate(int s);
  bool init();
  bool isSoundOK() {return soundOK;}
  bool listCards();
  void run();
  bool startCapture();
  bool startPlayback();
  bool startCalibration(bool capture);
  double getPlaybackStartupTime();
  int play();
  int capture();
  int captureCalibration();
  int playbackCalibration();
  void idle();
  void stopAndWait();
  bool isCapturing() {return captureState!=CPINIT;}
  bool stoppedPlaying();
  void abortPlayback() {playbackState=PBINIT;}
  QString *getLastError() {return &lastError;}
  QStringList *getPCMNamList(bool input)
    {
    if (input) return &inputPCMNameList;
    else return &outputPCMNameList;
    }
  QString inputAudioDevice; //!< audio device (e.g. /dev/dsp)
  QString outputAudioDevice; //!< audio device (e.g. /dev/dsp)
  edataSrc soundRoutingInput;
  edataDst soundRoutingOutput;
  uint recordingSize;
  buffer<short int,24> rxBuffer;
  buffer<SOUNDFRAME,PERIODBITS+5> txBuffer;
  short int tempRXBuffer[2*PERIODSIZE]; // twice because we are using stereo input
  wavIO waveIn;
  wavIO waveOut;
  uint frameCounter;
  void setStereo(bool st) {isStereo=st;}
  int calibrationCount(int &frames);
//  {
//    int tempT,tempC;
//    mutex.lock();
//    tempC=calibrationFrames;
//    tempT=calibrationTime;
//    mutex.unlock();
//    frames=tempC-CALIBRATIONLEADIN;
//    return tempT;
//  }

signals:
  void playbackStopped();


private:
  bool initialized;
  snd_pcm_t      *playbackHandle;
  snd_pcm_t      *captureHandle;
  QStringList inputPCMNameList; /**< list of PCM Names for input*/
  //QStringList inputPCMInfoList; /**< list of PCM Descriptions for input*/
  QStringList outputPCMNameList; /**< list of PCM Names for output*/
  //QStringList outputPCMInfoList; /**< list of PCM Descriptions for output*/
  QString lastError;
  bool errorHandler(int err,QString info);
  void errorHandler(QString info);
  bool setupSoundParams(bool isCapture);
  int samplerate;
  snd_pcm_uframes_t playbackPeriodSize;
  snd_pcm_uframes_t playbackBufferSize;
  snd_pcm_uframes_t capturePeriodSize;
  snd_pcm_uframes_t captureBufferSize;
  snd_pcm_hw_params_t *hwparams;
  snd_pcm_sw_params_t *swparams;
  bool abort;
  unsigned long storedFrames;
  eplaybackState playbackState;
  eplaybackState detectedPlaybackState;
  ecaptureState  captureState;
  ecaptureState  detectedCaptureState;
  bool soundOK;

  unsigned int minChannelsCapture;
  unsigned int maxChannelsCapture;
  unsigned int minChannelsPlayback;
  unsigned int maxChannelsPlayback;
  bool isStereo;
  void getDevices(snd_ctl_t *cardHandle, int cardNum, int devNum, snd_pcm_info_t *pcmInfo, snd_pcm_stream_t direction);
  QTime stopwatch;
  QMutex mutex;
  unsigned int calibrationFrames;
  int calibrationTime;

};

/*! \var soundIO *soundIOPtr
\brief predefined pointer to soundIO, must be initialized by the main program
*/
extern soundIO *soundIOPtr;

#endif // SOUNDIO_H
